package explorer;

import cz.cuni.pogamut.MessageObjects.Triple;
import java.io.File;
import java.io.IOException;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Set;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/** Unreal Tournament 2004 Map data structure.
 * 
 * This is data structure for UT2004 map. Map can be loaded from XML file
 * which is output of Explorer bot. The only useful method implemeted
 * there is Dijkstra algorithm for finding shortest path. Because this map
 * is basically directed graph, the structure is adjacency list.
 * 
 * @author Jiri Machalek
 * 2008-05-02
 */
public class UT2004Map {
    
    /** Class representing vertex in graph with list of edges. */
    static class Vertex {
        public List<Edge> neighbours = null;
        public Triple location = null;
        public String UnrealID = null;
    }
    
    /** Class representing edge in graph with list of jumps.
     * 
     * Note: The vertex edgeEnd in this class haven't filled list of neighbours.
     */
    static class Edge {
        public Vertex edgeEnd = null;
        public List<Triple> jumps = null;
    }

    /** Vertices in graph mapped by their UnrealID. */
    private Map<String,Vertex> vertices = null;
    
    /** Creates a new instance of map. */
    private UT2004Map() {
        vertices = new HashMap<String,Vertex>();
    }
    
    /** XML map-file parser creating new instance of UT2004Map. */
    public static UT2004Map loadFromXML(String filename) {
        final UT2004Map map = new UT2004Map();
        
        class MyHandler extends DefaultHandler {
            public Vertex workVertex = null;
            public Edge workEdge = null;

            @Override
            public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
                super.startElement(uri, localName, qName, attributes);
                //System.out.println(qName);
                if(qName.equals("NavPoint")) {
                    workVertex = new Vertex();
                    workVertex.UnrealID = attributes.getValue("UnrealID");
                    workVertex.location = new Triple();
                    workVertex.location.x = Double.parseDouble(attributes.getValue("LocationX"));
                    workVertex.location.y = Double.parseDouble(attributes.getValue("LocationY"));
                    workVertex.location.z = Double.parseDouble(attributes.getValue("LocationZ"));
                    workVertex.neighbours = new LinkedList<Edge>();
                }
                if(qName.equals("NeighbouringNavPoint")) {
                    workEdge = new Edge();
                    workEdge.jumps = new LinkedList<Triple>();
                    workEdge.edgeEnd = new Vertex();
                    workEdge.edgeEnd.UnrealID = attributes.getValue("UnrealID");
                    workEdge.edgeEnd.location = new Triple();
                    workEdge.edgeEnd.location.x = Double.parseDouble(attributes.getValue("LocationX"));
                    workEdge.edgeEnd.location.y = Double.parseDouble(attributes.getValue("LocationY"));
                    workEdge.edgeEnd.location.z = Double.parseDouble(attributes.getValue("LocationZ"));
                    workEdge.edgeEnd.neighbours = null;
                }
                if(qName.equals("Jump") && (workEdge != null) && (workEdge.jumps != null)) {
                    Triple workTriple = new Triple();
                    workTriple.x = Double.parseDouble(attributes.getValue("LocationX"));
                    workTriple.y = Double.parseDouble(attributes.getValue("LocationY"));
                    workTriple.z = Double.parseDouble(attributes.getValue("LocationZ"));
                    workEdge.jumps.add(workTriple);
                }
            }

            @Override
            public void endElement(String uri, String localName, String qName) throws SAXException {
                super.endElement(uri, localName, qName);
                //System.out.println("/" + qName);
                if(qName.equals("NavPoint")) {
                    map.vertices.put(workVertex.UnrealID, workVertex);
                }
                if(qName.equals("NeighbouringNavPoint") && (workVertex != null) && (workVertex.neighbours != null)) {
                    workVertex.neighbours.add(workEdge);
                }
            }
        }

        try {
            SAXParserFactory factory = SAXParserFactory.newInstance();
            factory.newSAXParser().parse(new File(filename), new MyHandler());
        } catch (SAXException e) {
            System.out.println("SAXException");
        } catch (ParserConfigurationException e) {
            System.out.println("ParserConfigurationException");
        } catch (IOException e) {
            System.out.println("IOException");
        }
        
        return map;
    }
    
    /** Some examples how to use this class. */
    public static void main(String[] args) {
        UT2004Map map = loadFromXML("c:\\Program Files\\NetBeans 6.0.1\\DM-Flux2.xml");
        System.out.println("Loaded map:");
        map.printMap();
        
        List<Triple> path = map.findWay("DM-Flux2.InventorySpot94", "DM-Flux2.PathNode46");
        if(path != null) {
            System.out.println("PATH from DM-Flux2.PathNode87 to DM-Flux2.PathNode46:");
            for(Triple tr : path) System.out.println("\t" + tr.toString());
        } else System.out.println("No known PATH.");
    }
    
    /** Straightforward map printing. */
    public void printMap() {
        for(Vertex vr : vertices.values()) {
            System.out.println(vr.UnrealID + " (" + vr.location.toString() + "):");
            for(Edge ed : vr.neighbours) {
                System.out.println("\t" + ed.edgeEnd.UnrealID + " (" + ed.edgeEnd.location.toString() + ")");
                for(Triple tr : ed.jumps) {
                    System.out.println("\t\tJump at " + tr.toString());
                }
            }
        }
    }
    
    /** Finding shortest path with Dijkstra algorithm.
     * 
     * Returns list of directly accessible triples which have to be followed to
     * reach given target. It means that this list is empty if the source is the
     * same as the target. If there is no path method returns null!
     */
    public List<Triple> findWay(String source, String target) {
        /** Class implementing Dijkstra algorithm adjusted for this data structure. */
        class Dijkstra {
            /** Set of settled nodes. */
            Set<String> settledNodes = new HashSet<String>();
            /** Map of predecessors for each vertex in shortest path from source. */
            Map<String,String> predecessors = new HashMap<String,String>();
            /** Map of shortest distances for each vertex. */
            Map<String, Double> shortestDistances = new HashMap<String, Double>();

            /** Comparator comparing distances from source. */
            private final Comparator<String> shortestDistanceComparator = new Comparator<String>() {
                @Override
                public int compare(String left, String right) {
                    double shortestDistanceLeft = getShortestDistance(left);
                    double shortestDistanceRight = getShortestDistance(right);
                    
                    if(shortestDistanceLeft > shortestDistanceRight) {
                        return +1;
                    }
                    else if(shortestDistanceLeft < shortestDistanceRight) {
                        return -1;
                    } else { // equal
                        return left.compareTo(right);
                    }
                }
            };

            /** Priority queue using shortestDistanceComparator. */
            private final PriorityQueue<String> unsettledNodes = new PriorityQueue<String>(100, shortestDistanceComparator);

            /** Search initilization. */
            private void initDijkstra(String start) {
                settledNodes.clear();
                unsettledNodes.clear();
                predecessors.clear();
                
                for(Vertex v : vertices.values()) {
                    setShortestDistance(v.UnrealID, Double.MAX_VALUE);
                }
                setShortestDistance(start, 0);
                unsettledNodes.add(start);
            }

            /** Standard relax procedure. */
            private void relaxNeighbors(String u) {
                Vertex v = vertices.get(u);
                if(v != null) {
                    for(Edge e : v.neighbours) {
                        if(getShortestDistance(e.edgeEnd.UnrealID) > (getShortestDistance(v.UnrealID)+Triple.distanceInSpace(v.location, e.edgeEnd.location))) {
                            setShortestDistance(e.edgeEnd.UnrealID, getShortestDistance(v.UnrealID)+Triple.distanceInSpace(v.location, e.edgeEnd.location));
                            unsettledNodes.add(e.edgeEnd.UnrealID);
                            predecessors.put(e.edgeEnd.UnrealID, v.UnrealID);
                        }
                    }
                }
            }
            
            /** Setter for shortest distance. */
            private void setShortestDistance(String vertex, double distance)
            {
                shortestDistances.put(vertex, distance);
            }

            /** Getter for shortest distance. */
            public double getShortestDistance(String vertex)
            {
                Double d = shortestDistances.get(vertex);
                return (d == null) ? Double.MAX_VALUE : d;
            }

            /** Min item extract from queue unsettledNodes. */
            private String extractMin() {
                return unsettledNodes.poll();
            }
            
            /** Main search method.
             *
             * Returned list is described above.
             */
            public List<Triple> execute(String start, String destination)
            {
                initDijkstra(start);

                while (!unsettledNodes.isEmpty())
                {
                    // get the node with the shortest distance
                    String u = extractMin();
                    // destination reached, stop
                    if (u.equals(destination)) break;
                    settledNodes.add(u);
                    relaxNeighbors(u);
                }
                
                String iter = destination;
                List<Triple> path = new LinkedList<Triple>();
                while((iter != null) && (!iter.equals(start))) {
                    if(!vertices.containsKey(iter)) return null;
                    path.add(0, vertices.get(iter).location);
                    iter = predecessors.get(iter);
                }
                return iter == null ? null : path;
            }
        }
        
        Dijkstra dijk = new Dijkstra();
        
        return dijk.execute(source, target);
    }
}
